
#ifndef CRYS_AES_CF_H
#define CRYS_AES_CF_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "DX_VOS_BaseTypes.h"
#include "CRYS_error.h"
#include "CRYS_Defs.h"
#include "LLF_AES_EngineInfo.h"
#include "CRYS_AES.h"

#ifdef __cplusplus
extern "C"
{
#endif

  /*
   *  Object % CRYS_AES_CF.h    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 16:42:30 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This file contains all of the enums and definitions that are used for the 
   *         CRYS AES APIs, as well as the APIs themselves. 
   *
   *  \version CRYS_AES.h#1:incl:13
   *  \author adams
   *  \remarks Copyright (C) 2005 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************************ Defines ******************************/


/************************ Enums ********************************/


/************************ Typedefs  ****************************/


/************************ context Structs  ******************************/


/************************ Public Variables **********************/


/************************ Public Functions **********************/

/* ------------------------------------------------------------
 * @brief This function is used to initialize the AES Skip parameters.
 *        In order to operate the AES CTR mode with skip these parameters must be 
 *        set before the AES machine is configure..
 *
 *
 *        The function executes the following major steps:
 *
 *        1. Validates all of the inputs of the function. If one of the received 
 *           parameters is not valid it shall return an error:
 *
 *           - verifying that the context pointer is not DX_NULL (*ContextID_ptr).
 *           - verifying Skip page size ( 0 < pageSize <=0xFFF ).
 *           - verifying Skip first count ( smaller than page size).
 *           - verifying Skip count (smaller than page size).
 *        2. Decrypting the received context to the working context after capturing 
 *           the working context by calling the CRYS_CCM_GetContext() call.
 *
 *        3. Validate context parameters :
 *           - verifying that the context working mode is CTR.
 *
 *        4. set skip parameters :
 *           - call LLf function.
 *
 *        4. Encrypting the information in the working context and storing 
 *           it to the users received context. After then the working context is released.
 *           This state is operated by calling the CRYS_CCM_EncryptAndReleaseContext call.
 *
 *        5. Exit the handler with the OK code.
 *
 *      
 *
 * @param[in] ContextID_ptr - a pointer to the AES context buffer allocated by the user that
 *                       is used for the AES machine operation.
 *
 * @param[in] SkipEnable - 
 *
 * @param[in] SkipPageSize -  
 *
 * @param[in] SkipFirstCnt - 
 *
 * @param[in] SkipCnt - 
 *
 * @return CRYSError_t - 
 */

CRYSError_t  CF_CRYS_AES_SetSkip( 	CRYS_AESUserContext_t    *ContextID_ptr,
                           			DxBool_t	SkipEnable,
                           			DxUint32_t	SkipPageSize,
			                        DxUint32_t	SkipFirstCnt,
			                        DxUint32_t  SkipCnt);


/* ------------------------------------------------------------
 *
 * @brief This function is used to activate the AES machine without starting the encryption.
 *        This function should be called after the CRYS_AES_Init or 
 *        CRYS_AES_InitSecretKey function was called.  
 *		  This function uses only DMA and not direct accesss memory.      
 *        The function should be called as part of procedure:
 *      
 *			CF_CRYS_AES_Config_Eng();
 *			CF_CRYS_AES_Execute();
 *			CF_CRYS_AES_Execute_End();
 *
 *		  and not by itself.
 *
 * @param[in] ContextID_ptr - A pointer to the AES context buffer allocated by the user 
 *                       that is used for the AES machine operation. This should be the 
 *                       same context as was used for the previous call of this session.
 *
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * @param[in] DataInSize - The size of the input data.
 *
 * @param[in,out] DataOut_ptr - The pointer to the buffer of the output data from the AES. 
 *                        The pointer's value does not need to be word-aligned.  
 *
 * @return CRYSError_t - On success the value CRYS_OK is returned, 
 *                        and on failure a value from CRYS_AES_error.h
 */
 CRYSError_t  CF_CRYS_AES_ConfigEng( CRYS_AESUserContext_t       *ContextID_ptr,   
                              		  DxUint8_t                     *DataIn_ptr,     
                              		  DxUint32_t                    DataInSize,     
                              		  DxUint8_t                     *DataOut_ptr ); 



/* ------------------------------------------------------------
 *
 * @brief This function is used to activate DMA after the AES engine
 *        was set previously.
 *        This function MUST be call as part of the procedure:
 *
 *			CF_CRYS_AES_Config_Eng();
 *			CF_CRYS_AES_Execute();
 *			CF_CRYS_AES_Execute_End();
 *
 *        The function does not get context and there for it just call LLF:
 *
 *        1. Activate DMA:
 *           - call LLf function.
 *
 *        2. Exit the handler with the OK code.
 *
 *      
 *
 *
 *
 * @return CRYSError_t - 
 */

CRYSError_t  CF_CRYS_AES_Execute(void);


/* ------------------------------------------------------------
 *
 * @brief This function is used to end the AES processing session.
 *        The function shuts down the AES engine.
 *      
 *        The function executes the following major steps:
 *
 *        1. Decrypting the received context to the working context after capturing 
 *           the working context by calling the CRYS_CCM_GetContext() call.
 *
 *        2. Validate context parameters :
 *         
 *
 *        3. Finish all AES processing:
 *           - call LLf function.
 *
 *        4. Encrypting the information in the working context and storing 
 *           it to the users received context. After then the working context is released.
 *           This state is operated by calling the CRYS_CCM_EncryptAndReleaseContext call.
 *
 *        5. Exit the handler with the OK code.
 *
 * @param[in] ContextID_ptr - A pointer to the AES context buffer allocated by the user that
 *                       is used for the AES machine operation. This should be the 
 *                       same context as was used for the previous call of this session.
 *
 * @param[in,out] DataOut_ptr - The pointer to the buffer of the output data from the AES. 
 *                        The pointer's value does not need to be word-aligned.  
 *						  (This parameter is needed only for the AES MAC Mode where
 *							we copy the result right from the DOUT with out using ODMA).
 *
 * @return CRYSError_t - On success the value CRYS_OK is returned, 
 *                        and on failure a value from CRYS_AES_error.h
 */
 CRYSError_t  CF_CRYS_AES_ExecuteEnd(CRYS_AESUserContext_t       *ContextID_ptr,   
                              		  DxUint8_t                     *DataIn_ptr,     
                              		  DxUint32_t                    DataInSize,     
                              		  DxUint8_t                     *DataOut_ptr );



#ifdef __cplusplus
}
#endif

#endif
